/*:
 * @target MZ
 * @plugindesc 初回起動だけボイス音量を指定値に初期化して保存（その後はユーザー設定を尊重）
 *
 * @param defaultVoiceVolume
 * @text 初回起動のボイス音量
 * @type number
 * @min 0
 * @max 100
 * @default 60
 *
 * @help
 * - 初回判定は config ではなく、独自の保存ファイル（meg_voice_init_once）で行います。
 * - 初回だけタイトル画面の最初の更新でボイス音量を設定し、ConfigManager.save()で保存します。
 * - 2回目以降は一切上書きしません。
 */

(() => {
  "use strict";

  const script = document.currentScript;
  const pluginName = script.src.match(/([^/]+)\.js$/)[1];
  const params = PluginManager.parameters(pluginName);

  const DEFAULT_VOL = Math.max(0, Math.min(100, Number(params.defaultVoiceVolume ?? 60)));

  // 初回判定用：configとは別の保存キー（save/meg_voice_init_once.rmmzsave）
  const INIT_KEY = "meg_voice_init_once";

  // どのプラグインに上書きされても、保存データに voiceVolume を含める保険
  const _makeData = ConfigManager.makeData;
  ConfigManager.makeData = function() {
    const data = _makeData.call(this);
    if (Object.getOwnPropertyDescriptor(ConfigManager, "voiceVolume")) {
      data.voiceVolume = ConfigManager.voiceVolume;
    }
    return data;
  };

  // 初回フラグの読み込み（非同期）
  let initChecked = false;
  let isFirstBoot = false;
  const initPromise = StorageManager.loadObject(INIT_KEY)
    .then(obj => {
      initChecked = true;
      isFirstBoot = !(obj && obj.done === true);
    })
    .catch(() => {
      initChecked = true;
      isFirstBoot = true;
    });

  function setVoiceVolume(vol) {
    if (Object.getOwnPropertyDescriptor(ConfigManager, "voiceVolume")) {
      ConfigManager.voiceVolume = vol; // SimpleVoice の setter 経由で AudioManager も更新される想定
      return true;
    }
    if (Object.getOwnPropertyDescriptor(AudioManager, "voiceVolume")) {
      AudioManager.voiceVolume = vol;
      return true;
    }
    // 最後の保険
    AudioManager._voiceVolume = vol;
    return false;
  }

  let applied = false;

  // タイトル画面の「最初の1フレーム」で適用（他プラグインのTitle.start後に勝てる）
  const _SceneTitle_update = Scene_Title.prototype.update;
  Scene_Title.prototype.update = function() {
    _SceneTitle_update.call(this);

    if (applied) return;
    if (!initChecked) return; // 初回判定の読み込み待ち

    if (!isFirstBoot) {
      applied = true;
      return;
    }

    // 初回だけ適用 → 保存
    setVoiceVolume(DEFAULT_VOL);

    // 設定保存（= 次回以降ユーザー設定が維持される）
    ConfigManager.save();

    // 初回フラグ保存（configとは別）
    StorageManager.saveObject(INIT_KEY, { done: true, v: DEFAULT_VOL, t: Date.now() });

    applied = true;
  };

  // 起動が速すぎて update より前に initChecked が立たないケースを避ける
  initPromise.catch(() => {});
})();

